/* 
 * FilePollingThread.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.pollingthreads;

import de.esukom.decoit.ifmapclient.main.IfMapClient;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;

/**
 * Abstract base Class for reading and polling files
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public abstract class FilePollingThread extends PollingThread {

    // properties from config file
    public String filePath;

    // (snort-log)-file to watch/poll
    public File file;

    // flag for indicating that reading of file is necessary, even if there
    // was no update (e.g. at startup)
    public boolean isFirstStart = false;

    // is used to determine position of last read-in entries, so we
    // can skip the parsing of already parsed events in next cycle
    // (-> when file was updated)
    public int lastEntryLineNumber = 0;

    // time stamp of last file modification, is used to determine changes
    // of the file
    public long lastModified;

    @Override
    protected void initProperties(Properties props) {

        // get file-path from configuration
        filePath = props.getProperty("filepath", null);
        if (filePath == null || filePath.length() == 0) {
            IfMapClient
                    .exit("error while initializing SnortBarnyardFilePolling Thread - file path cannot be null or empty, please check your configuration");
        }

        // flag indicating first start
        isFirstStart = true;

        // open file
        IfMapClient.LOGGER.config("opening log-file at path: " + filePath + "...");

        if ((new File(filePath)).exists()) {
            file = new File(filePath);
        } else {
            IfMapClient.exit("log-file at " + filePath
                    + " doesnt exist...please check your configuration");
        }

        lastModified = file.lastModified();
    }

    @Override
    public void run() {
        while (running) {
            if (!pausing) {
                try {
                    long actualLastModified = file.lastModified();
                    // check if file changes have occurred since last cycle
                    IfMapClient.LOGGER.info("checking log-file  " + filePath + " for updates...");
                    if (isFirstStart | lastModified != actualLastModified) {
                        lastModified = actualLastModified;
                        IfMapClient.LOGGER.info("log-file " + filePath
                                + " has been updated, looking for new entries...");
                        notify(readFile());
                    } else {
                        IfMapClient.LOGGER
                                .info("log-file " + filePath + " has not been updated...");
                    }
                    Thread.sleep(sleepTime);
                } catch (Exception e) {
                    e.printStackTrace();
                    IfMapClient.exit("error while checking log-file  " + filePath + " for updates");
                }
            }
        }
    }

    @Override
    public void notify(Object o) {
        // hold Thread
        if (o != null) {
            ArrayList<HashMap<String, String>> resultList = (ArrayList<HashMap<String, String>>) o;
            if (resultList.size() > 0) {
                pausing = true;
                setChanged();
                notifyObservers(resultList);
            } else {
                IfMapClient.LOGGER.info("no new entries in  " + filePath
                        + "  log-file, not calling observer");
            }
        } else {
            IfMapClient.LOGGER.info("retrieved update in  " + filePath
                    + " log-file is null, not calling observer");
        }
    }

    /**
     * read entries from file and save them in String-HashMap
     * 
     * 
     * @return
     */
    protected abstract ArrayList<HashMap<String, String>> readFile();
}