/* 
 * IPTablesULogFilePollingThread.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.pollingthreads;

import de.esukom.decoit.ifmapclient.config.GeneralConfig;
import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Matcher;

/**
 * Thread for polling the IP-Tables-ULOG-File for new IPTables-Events
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class IPTablesULogFilePollingThread extends FilePollingThread {

    // ip-blacklist from configuration
    // ip-addresses contained here will
    // not be processed and therefore will
    // be forever "blocked"
    private String[] mIpBlacklist;

    // log-skipped-entries-flag (read from file-properties-config)
    private boolean mLogSkippedEntries;

    // for mapping alphanumeric month-values from ulog to numeric values
    private HashMap<String, String> mMonthValues;

    /**
     * constructor
     * 
     * @param path
     *            path of snort-log-file
     */
    public IPTablesULogFilePollingThread(Properties pr) {
        // initialize properties
        initProperties(pr);
    }

    @Override
    protected void initProperties(Properties props) {
        super.initProperties(props);

        // initialize month-alphanumeric to month-numeric map
        initMonthMap();

        // ip-blacklist
        mIpBlacklist = props.getProperty("iptables.iplist.blacklist").split(",");
        IfMapClient.LOGGER.config("reading in ip-blacklist from enforcement.properties...");
        if (mIpBlacklist == null || mIpBlacklist.length < 1) {
            IfMapClient.LOGGER.config("no ip-addresses defined in configuration for blacklist");
        }
        for (int i = 0; i < mIpBlacklist.length; i++) {
            IfMapClient.LOGGER.config("entry [" + i + "] on blacklist has ip-address: "
                    + mIpBlacklist[i]);
        }

        // log skipped entries ?
        mLogSkippedEntries = Toolbox.getBoolPropertyWithDefault("iptables.log.skippedentries",
                false, props);
    }

    /**
     * read and parse alert-log-file
     * 
     * @return list containing all read in entries from ulog-file
     */
    @Override
    protected ArrayList<HashMap<String, String>> readFile() {
        // used for distinction of the different lines of a single
        // event entry inside log-file while parsing
        int currentLineInSingleEntry = 0;

        // number of read lines in current cycle, used to determine
        // current position inside file (in terms of already read lines )
        int currentCycleLineNumber = 0;

        BufferedReader input = null;
        ArrayList<HashMap<String, String>> result = new ArrayList<HashMap<String, String>>();

        HashMap<String, String> tempEventData = new HashMap<String, String>();

        try {
            input = new BufferedReader(new FileReader(file), 1);
            String line = null;
            String entryDate = null;
            boolean skipCurrentEntry = false;

            while ((line = input.readLine()) != null) {
                if (isFirstStart) {
                    lastEntryLineNumber++;
                }

                currentCycleLineNumber++;
                if (isFirstStart | currentCycleLineNumber > lastEntryLineNumber) {
                    if (!skipCurrentEntry) {
                        // convert last entry
                        if (!tempEventData.isEmpty()) {
                            // clone! otherwise you will get a reference-problem
                            // after clearing tempEventData
                            result.add((HashMap<String, String>) tempEventData.clone());
                        }
                    } else {
                        // reset skip-current-entry-flag
                        skipCurrentEntry = false;
                    }

                    // reset entries from last cycle
                    tempEventData.clear();

                    // reset current entry line counter
                    currentLineInSingleEntry = 0;

                    // get ip of current entry and perform blacklist-check
                    // against the parsed in ip-address
                    Matcher ip4Matcher = Toolbox.getRegExPattern(Toolbox.REGEX_GENERAL_IP4)
                            .matcher(line);
                    if (ip4Matcher.find()) {
                        if (inBlacklist(ip4Matcher.group())) {
                            skipCurrentEntry = true;
                            if (mLogSkippedEntries) {
                                IfMapClient.LOGGER.warning("current entry is inside ip-blacklist ["
                                        + ip4Matcher.group() + "]...will now skip this entry");
                            }

                        }
                    } else {
                        skipCurrentEntry = true;
                        IfMapClient.LOGGER
                                .warning("could not find valid IP4-address in current entry...will now skip this entry");
                    }

                    if (!skipCurrentEntry) {
                        // get timestamp of current entry
                        Matcher timestampMatcher = Toolbox.getRegExPattern(
                                Toolbox.REGEX_IPTABLES_ULOG_TIMESTAMP).matcher(line);
                        if (timestampMatcher.find()) {
                            // convert entry to "ifmap-compatible" format
                            // to perform date check
                            entryDate = rearrangeDate(Toolbox.getCurrentYear() + "/"
                                    + timestampMatcher.group());

                            // replace current entries date with new format
                            // so we don't need to do it again inside the
                            // mapping factory
                            String newLine = line.replaceFirst(
                                    Toolbox.REGEX_IPTABLES_ULOG_TIMESTAMP, entryDate);
                            line = newLine;
                        } else {
                            skipCurrentEntry = true;
                            IfMapClient.LOGGER
                                    .warning("could not find timestamp in current entry...will now skip this entry");
                        }

                        // if "sendold"-option is disabled, skip entries which
                        // time-stamp is before the clients startup-time. Will
                        // be done here (instead of Mapping-Factory) because
                        // the ulog-file can really grow big an we don't want to
                        // pass all these non-used entries through the mapping
                        // factory due to performance reasons
                        if (!GeneralConfig.APPLICATION_MESSAGING_SENDOLD) {
                            if (Toolbox.getCalendarFromString(entryDate).after(
                                    Toolbox.getCalendarFromString(Toolbox.sClientStartTime))) {

                                // add current entry
                                tempEventData.put(new Integer(currentLineInSingleEntry).toString(),
                                        line);

                                currentLineInSingleEntry++;
                            }
                        } else {
                            // add current entry without time-stamp-check
                            tempEventData.put(new Integer(currentLineInSingleEntry).toString(),
                                    line);
                        }
                    }
                }
            }

            // loop over, add last remaining entry
            if (!tempEventData.isEmpty() & !skipCurrentEntry) {
                result.add(tempEventData);
            }

            if (!isFirstStart) {
                // set new last entry index
                lastEntryLineNumber = currentCycleLineNumber;
            } else {
                // first cycle finished
                isFirstStart = false;
            }
        } catch (FileNotFoundException ex) {
            ex.printStackTrace();
            IfMapClient
                    .exit("error while opening ip-tables ulog-file...the file does not exists, please check your configuration");
        } catch (IOException ex) {
            ex.printStackTrace();
            IfMapClient.exit("I/O error while reading iptables ulog-file");
        } finally {
            try {
                input.close();
            } catch (IOException e) {
                e.printStackTrace();
                IfMapClient
                        .exit("error while closing input buffer after trying to read iptables ulog-file");
            }
        }

        return result;
    }

    /**
     * initialize month-mapping-map for ip-tables-ulog-file
     */
    private void initMonthMap() {
        mMonthValues = new HashMap<String, String>();
        mMonthValues.put("Jan", "01");
        mMonthValues.put("Feb", "02");
        mMonthValues.put("Mar", "03");
        mMonthValues.put("Apr", "04");
        mMonthValues.put("May", "05");
        mMonthValues.put("Jun", "06");
        mMonthValues.put("Jul", "07");
        mMonthValues.put("Aug", "08");
        mMonthValues.put("Sep", "09");
        mMonthValues.put("Oct", "10");
        mMonthValues.put("Nov", "11");
        mMonthValues.put("Dec", "12");
    }

    /**
     * rearrange passed in date (as parsed from alertlog-file) to fit IF-MAP
     * Timestamp format
     * 
     * @param String
     *            currentDate [YYYY/MONTH(3-chars)_DD_HH:MM:SS]
     * 
     * @return timestamp in IF-MAP format
     */
    private String rearrangeDate(String currentDate) {
        String[] date = new String[3];
        date[0] = currentDate.substring(0, 4); // Year
        date[1] = mMonthValues.get(currentDate.substring(5, 8)); // Month

        // Day of Month is kind of ugly in ulog-file: if the date consist only
        // of one digit, u-log doesn't fill it up with a leading zero (e.g
        // 02.08.2011),
        // instead it leaves a space (e.g. 2.08.2011). So we need to consider
        // this "special case"
        date[2] = currentDate.substring(9, 11); // Day
        if (date[2].startsWith(" ")) {
            date[2] = date[2].replaceFirst(" ", "0");
        }

        // YYYY/MM/DD-hh:mm:ss.S => [0]Date [1]Time
        String timestamp = currentDate.substring(12, currentDate.length()); // Day

        // return new timestamp-string
        return date[0] + "-" + date[1] + "-" + date[2] + " " + timestamp;
    }

    /**
     * check if passed in ip-address-string is contained inside ip-blacklist
     * 
     * @param ip
     *            ip-address to be checked
     * 
     * @return true, if ip-address inside blacklist
     */
    private boolean inBlacklist(String ip) {
        for (int i = 0; i < mIpBlacklist.length; i++) {
            if (mIpBlacklist[i].equals(ip)) {
                return true;
            }
        }
        return false;
    }
}