/* 
 * NagiosSocketPollingThread.java        0.1.4 12/02/16
 *  
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.pollingthreads;

import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;
import java.util.Scanner;

/**
 * Thread for listening and receiving nagios-events from a specified port
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class NagiosSocketPollingThread extends PollingThread {

    // properties from config file
    private int mNagiosPort;

    // Socket to listen to
    private ServerSocket mProviderSocket;

    // Socket specific fields
    private Socket mConnection = null;
    private BufferedReader mBufferReader = null;
    private DataInputStream mIn = null;

    /**
     * constructor
     * 
     * @param path
     *            path of snort-log-file
     */
    public NagiosSocketPollingThread(Properties pr) {

        // initialize properties
        initProperties(pr);

        IfMapClient.LOGGER.config("initalizing socket for nagios messages which listens on port: "
                + mNagiosPort);
        try {
            mProviderSocket = new ServerSocket(mNagiosPort);
        } catch (IOException e) {
            e.printStackTrace();
            IfMapClient.exit("I/O error while creating server socket for nagios polling-thread");
        }
    }

    @Override
    protected void initProperties(Properties props) {
        mNagiosPort = new Integer(props.getProperty("nagios.server.port", "0")).intValue();
        if (mNagiosPort <= 0) {
            IfMapClient
                    .exit("error while initializing NagiosSocketPolling Thread - property nagiosPort must be above zero");
        }
    }

    /**
     * parse a single line
     * 
     * @param line
     *            the line that needs to be parsed
     * 
     * @return 2d-list containing the different nagios macros
     */
    private HashMap<String, String> parseLine(String line) {
        /*
         * MAKRONAME:VALUE;MAKRONAME:VALUE;MAKRONAME:VALUE;... ONE LINE!
         * 
         * possible makro-values for hosts: hostname, hostalias, hostaddress,
         * hoststate, hoststatetype, hostoutput
         * 
         * possible makro-values for services:
         * 
         * servicestate, servicestatetype, serviceattempt, servicedescription
         * serviceoutput, servicelatency, serviceduration, servicedowntime
         * servicenotes
         */
        // check if string is "valid"
        // current validation rule: minimum of two chars seperated by ":" and
        // ending with ";"
        if (line != null && line.matches(Toolbox.REGEX_NAGIOS_PORTPOLLER_ISSTRINGVALID)) {
            IfMapClient.LOGGER.fine("incomming line to parse: " + line);
            // initialize temporary nagios-makro-list
            HashMap<String, String> makrosList = new HashMap<String, String>();

            // get "makro:value" pair as array
            Scanner scanner = new Scanner(line);
            scanner.useDelimiter(";");
            while (scanner.hasNext()) {
                String pair = scanner.next();
                String[] parsedpair = pair.split("\\=");
                makrosList.put(parsedpair[0], parsedpair[1]);
            }
            return makrosList;
        } else {
            IfMapClient.LOGGER.warning("nagios polling-thread detected a unparsable line: " + line);
            return null;
        }
    }

    @Override
    public void run() {
        while (running) {
            if (!pausing) {
                try {
                    // wait for connection
                    mConnection = mProviderSocket.accept();

                    // I/O
                    mIn = new DataInputStream(mConnection.getInputStream());
                    mBufferReader = new BufferedReader(new InputStreamReader(mIn));

                    // read incoming line from server
                    String inputLine = "";
                    try {
                        inputLine = mBufferReader.readLine();
                        if (inputLine != null) {
                            IfMapClient.LOGGER.info("nagios polling-thread received message: "
                                    + inputLine + " from "
                                    + mConnection.getInetAddress().getHostName());
                            notify(parseLine(inputLine));
                        } else {
                            break;
                        }

                    } catch (IOException e) {
                        e.printStackTrace();
                        IfMapClient
                                .exit("nagios polling-thread detected an I/O error while reading from socket");
                    }
                } catch (IOException ioException) {
                    ioException.printStackTrace();
                    IfMapClient
                            .exit("nagios polling-thread encounters an error while building in/output-streams");
                } finally {
                    try {
                        mIn.close();
                        // providerSocket.close();
                    } catch (IOException ioException) {
                        ioException.printStackTrace();
                        IfMapClient
                                .exit("nagios polling-thread encounters an error while closing streams and socket");
                    }
                }
            }
        }
    }

    @Override
    public void notify(Object o) {
        if (o != null) {
            ArrayList<HashMap<String, String>> resultList = new ArrayList<HashMap<String, String>>();
            resultList.add((HashMap<String, String>) o);
            // in case of new entries, notify observer and pass the new entries
            if (resultList.size() > 0) {
                // notify the observer and pass new entries
                setChanged();
                notifyObservers(resultList);
            } else {
                // logically we should never reach this point...
                // so this is kind of a "just in case of..."
                IfMapClient.LOGGER
                        .info("no updates received from nagios-socket, not calling observer");
            }
        } else {
            IfMapClient.LOGGER
                    .warning("retrieved message from nagios-socket is empty, not calling observer");
        }
    }
}
