/* 
 * Toolbox.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.util;

import de.esukom.decoit.ifmapclient.logging.Logging;

import java.io.File;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/**
 * Class containing some Helper-Functions that doesn't fit anywhere else
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke
 */
public class Toolbox {

    // ***********************
    // * REGULAR EXPRESSIONS *
    // ***********************

    // general regular expressions used by different components
    public static final String REGEX_GENERAL_IP4 = "(([2]([0-4][0-9]|[5][0-5])|[0-1]?[0-9]?[0-9])[.]){3}(([2]([0-4][0-9]|[5][0-5])|[0-1]?[0-9]?[0-9]))";
    public static final String REGEX_GENERAL_IP6 = "((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])(.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9]?[0-9])){3}))|:)))(%.+)?\\s*";

    // regular expressions for nagios-component
    public static final String REGEX_NAGIOS_PORTPOLLER_ISSTRINGVALID = "((\\w+)=(.+);)+";

    // regular expressions for snort-component
    public static final String REGEX_SNORT_ALERTLOG_TYPE = "\\]\\s\\(?[\\w\\s]+\\)?\\:?[a-zA-Z0-9\\s\\<\\>\\/\\-\\*]*\\[";
    public static final String REGEX_SNORT_ALERTLOG_CLASS = "Classification:[\\w\\s]+";
    public static final String REGEX_SNORT_ALERTLOG_PRIORITY = "Priority:\\s\\d";
    public static final String REGEX_SNORT_ALERTLOG_TIMESTAMP = "\\d{2}\\/\\d{2}\\-\\d{2}\\:\\d{2}\\:\\d{2}\\.\\d{6}";
    public static final String REGEX_SNORT_ALERTLOG_VULNARIBILITY_URI = "\\[Xref.+\\]";

    // regular expressions for snort-portscan-component
    public static final String REGEX_SNORT_PORTSCANLOG_TIMESTAMP = "Time: "
            + REGEX_SNORT_ALERTLOG_TIMESTAMP;
    public static final String REGEX_SNORT_PORTSCANLOG_SOURCE_IP = REGEX_GENERAL_IP4 + "\\s->";
    public static final String REGEX_SNORT_PORTSCANLOG_TARGET_IP = "->\\s" + REGEX_GENERAL_IP4;
    public static final String REGEX_SNORT_PORTSCANLOG_SCANTYPE = "\\(portscan\\)\\s[\\w\\s]+";

    // regular expressions for iptables-component
    public static final String REGEX_IPTABLES_ENFORCEMENT_ATTRIBUTE = "(ip|name|significance|confidence|magnitude|message)";
    public static final String REGEX_IPTABLES_ALLOW_ATTRIBUTE = "(role_name|capability_name|device-characteristic_os-version|device-characteristic_os|device-characteristic_model|device-characteristic_manufacturer|device-characteristic_device-type|access-request-ip_ncp-ip-address-type)";
    public static final String REGEX_IPTABLES_FILTERSTRING_DELIMITERS = "\\[.*\\]";
    public static final String REGEX_IPTABLES_ULOG_TIMESTAMP = "\\w*\\s*\\d*\\s\\d*\\:\\d*:\\d*";
    public static final String REGEX_IPTABLES_IFMAP_TIMESTAMP = "\\d\\d\\d\\d\\-\\d\\d\\-\\d\\d\\s\\d\\d\\:\\d\\d\\:\\d\\d";

    // regular expressions for radius-component
    public static final String REGEX_RADIUS_DATE = "\\w{3}\\s\\w{3}\\s{1,2}\\d{1,2}\\s\\d{2}:\\d{2}:\\d{2}\\s\\d{4}";
    public static final String REGEX_RADIUS_LOGIN_DEFAULT = "Auth: Login \\w{2,9}(\\s\\(.*\\):)?";
    public static final String REGEX_RADIUS_USER_PW_DEFAULT = "(\\[\\w*(\\/\\w*)?\\])";
    public static final String REGEX_RADIUS_PORT_DEFAULT = "port\\s\\d{1,4}";
    public static final String REGEX_RADIUS_USER_DETAIL = "\\w{4}-\\w{4} = \".*\",";
    public static final String REGEX_RADIUS_ACCTSTATUS = "Acct-Status-Type = \\w*(-\\w*)?";
    public static final String REGEX_RADIUS_ACCTSESSIONID = "Acct-Session-Id = \"\\d*\"";
    public static final String REGEX_RADIUS_ACTT_UNIQUESESSIONID = "Acct-Unique-Session-Id = \"\\w*\"";

    // the logger
    public static Logger sLogger = Logging.getTheLogger();

    // application start time
    public static String sClientStartTime = now();

    /**
     * get regex-pattern for passed in regular-expression
     * 
     * @param regEx
     *            regular-expression-string to compile
     * 
     * @return regexPattern of the passed in expression-string
     */
    public static Pattern getRegExPattern(String regEx) {
        return Pattern.compile(regEx);
    }

    // ******************
    // * CONFIG-RELATED *
    // ******************

    /**
     * get string property value from Properties-Object
     * 
     * @param property
     *            property-name
     * @param conf
     *            properties-object
     * @param checkIfFileExists
     *            check the resulting string exists as a file. This is meant to
     *            be a helper function in case the passed in property is a
     *            file(path)...which is very common in our case
     * 
     * @return property as string
     */
    public static String getStringProperty(String property, Properties conf,
            boolean checkIfFileExists) {
        String propertyValue = conf.getProperty(property, null);
        if (propertyValue == null) {
            sLogger.warning("required property <" + property + "> not found");
            return null;
        } else if (propertyValue.length() == 0) {
            sLogger.warning("required property <" + property + "> cannot be emtpy");
            return null;
        } else {
            if (checkIfFileExists) {
                if ((new File(propertyValue)).exists()) {
                    sLogger.info("value for property <" + property + "> is: " + propertyValue);
                    return propertyValue;
                } else {
                    sLogger.warning("file for property  <" + property + "> at path "
                            + propertyValue + " doesnt exists");
                    return null;
                }
            } else {
                sLogger.info("value for property <" + property + "> is: " + propertyValue);
                return propertyValue;
            }
        }
    }

    /**
     * get string property value from Properties-Object, use passed in default
     * value if no property is found
     * 
     * @param property
     *            property-name
     * @param conf
     *            properties-object
     * @param default default-value that should be returned if property does not
     *        exists
     * 
     * @return property as string
     */
    public static String getStringPropertyWithDefault(String property, Properties conf,
            String defaultValue) {
        String propertyValue = conf.getProperty(property, null);
        if (propertyValue == null) {
            sLogger.warning("required property <" + property + "> not found...using default ("
                    + defaultValue + ")");
            return defaultValue;
        } else if (propertyValue.length() == 0) {
            sLogger.warning("required property <" + property
                    + "> cannot be emtpy......using default (" + defaultValue + ")");
            ;
            return defaultValue;
        } else {
            sLogger.info("value for property <" + property + "> is: " + propertyValue);
            return propertyValue;
        }
    }

    /**
     * get int property value from Properties-Object
     * 
     * @param property
     *            property-name
     * @param defaultValue
     *            default value in case without zero flag is set
     * @param conf
     *            properties-object
     * @param withoutZero
     *            in case the property is 0, return the passed in default value
     * 
     * @return property as int
     */
    public static int getIntPropertyWithDefault(String property, int defaultValue, Properties conf,
            boolean withoutZero) {
        int returnValue;
        String propertyValue = conf.getProperty(property, null);
        if (propertyValue == null || propertyValue.length() == 0) {
            sLogger.warning("property  <" + property + "> is empty or null...using default value ("
                    + defaultValue + ")");
            return defaultValue;
        }
        if (withoutZero) {
            if (propertyValue.equals("0")) {
                sLogger.warning("property  <" + property
                        + "> cannot be zero...using default value (" + defaultValue + ")");
                return defaultValue;
            }
        }
        try {
            returnValue = Integer.parseInt(propertyValue);
        } catch (NumberFormatException e) {
            sLogger.warning("property <" + property
                    + "> must be an integer...using default value (" + defaultValue + ")");
            return defaultValue;
        }

        sLogger.info("value for property <" + property + "> is: " + returnValue);

        return returnValue;
    }

    /**
     * get boolean property value from Properties-Object
     * 
     * @param property
     *            property-name
     * @param defaultValue
     *            default value in case the property is null/0
     * @param conf
     *            properties-object
     * 
     * @return property as boolean
     */
    public static boolean getBoolPropertyWithDefault(String property, boolean defaultValue,
            Properties conf) {
        boolean returnValue;
        String propertyValue = conf.getProperty(property, null);
        if (propertyValue == null || propertyValue.length() == 0) {
            sLogger.warning("property  <" + property + "> is empty or null...using default value ("
                    + defaultValue + ")");
            return defaultValue;
        }

        returnValue = Boolean.parseBoolean(propertyValue);
        sLogger.info("value for property <" + property + "> is: " + returnValue);

        return returnValue;
    }

    /**
     * check if array really contains values, if not set it to null. this is
     * necessary due to problems with reading the properties from config!
     * 
     * @param propArray
     *            array to check
     * 
     * @return boolean indicating wheter array is null
     */
    public static boolean checkConfig(String[] propArray) {
        if (propArray != null && propArray.length > 0) {
            if (propArray[0].length() == 0) {
                return false;
            }
        }
        return true;
    }

    // ****************
    // * DATE-RELATED *
    // ****************

    /**
     * get current time/date as string
     * 
     * @return current time/date as string
     */
    public static final String DATE_FORMAT_NOW = "yyyy-MM-dd HH:mm:ss";

    public static String now() {
        Calendar cal = Calendar.getInstance();
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT_NOW);
        return sdf.format(cal.getTime());
    }

    /**
     * get calendar-object from passed in string the format of the string has to
     * be: yyyy-MM-dd HH:mm:ss
     * 
     * @param dateString
     *            preset date of calendar
     * 
     * @return calendar-object set to passed in date
     */
    public static Calendar getCalendarFromString(String dateString) {
        DateFormat formatter;
        Date date = null;
        formatter = new SimpleDateFormat(DATE_FORMAT_NOW);
        try {
            date = (Date) formatter.parse(dateString);
        } catch (ParseException e) {
            System.err.println("Error while creating Calendar From String!");
            e.printStackTrace();
        }
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);

        return cal;
    }

    /**
     * get current year as string
     * 
     * @return String current year
     */
    public static String getCurrentYear() {
        return new Integer(new GregorianCalendar().get(GregorianCalendar.YEAR)).toString();
    }

    // **************
    // * OTHER/MISC *
    // **************

    /**
     * convert passed in ip6-address-string to IFMAP-compatible
     * ip6-address-string this means deleting all leading zeros from every
     * address-part
     * 
     * @param String
     *            adr address-string to convert
     * 
     * @return IFMAP-compatible ip6-address-string
     */
    public static String convertIP6AddressToIFMAPIP6AddressPattern(String adr) {
        String[] singleEntrys = adr.split(":");
        String convertedAddress = new String();

        for (int i = 0; i < singleEntrys.length; i++) {

            // delete "leading" zeros
            singleEntrys[i] = singleEntrys[i].replaceFirst("0*", "");

            // is string is empty after deleting leading zeros, add a zero-char
            if (singleEntrys[i].length() < 1) {
                singleEntrys[i] = "0";
            }

            // add ":" to address, leave out last entry
            if (i != singleEntrys.length - 1) {
                singleEntrys[i] += ":";
            }

            // add entry to converted-address-string
            convertedAddress += singleEntrys[i];
        }

        return convertedAddress;
    }

    public static void printLogo() {
        System.out.println(" +-----------------------------------------------------+");
        System.out.println(" |                   _                                 |");
        System.out.println(" |                  | |                                |");
        System.out.println(" |    ___  ___ _   _| | _____  _ __ ___                |");
        System.out.println(" |   / _ \\/ __| | | | |/ / _ \\| '_ ` _ \\               |");
        System.out.println(" |  |  __/\\__ \\ |_| |   < (_) | | | | | |              |");
        System.out.println(" |   \\___||___/\\__,_|_|\\_\\___/|_| |_| |_|              |");
        System.out.println(" |                                                     |");
        System.out.println(" |  DECOIT IF-MAP-CLIENT (VERSION 0.1.5)               |");
        System.out.println(" |  (c)2011 DECOIT GmbH (www.decoit.de)                |");
        System.out.println(" |                                                     |");
        System.out.println(" |  developed within the ESUKOM-Project(www.esukom.de) |");
        System.out.println(" |  licensed to the Apache Software Foundation (ASF)   |");
        System.out.println(" |                                                     |");
        System.out.println(" |  see README for more details                        |");
        System.out.println(" |                                                     |");
        System.out.println(" +-----------------------------------------------------+\n\n\n");
    }
}