/* 
 * SnortBarnyardFileEventMappingFactory.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.mappingfactory;

import de.esukom.decoit.ifmapclient.config.GeneralConfig;
import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.messaging.SOAPMessageSender;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import de.fhhannover.inform.trust.ifmapj.metadata.EventType;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Matcher;

/**
 * concrete implementation of abstract mapping-factory for mapping values from
 * polling-threads to event objects that can be send to MAP-Server
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class SnortBarnyardFileEventMappingFactory extends SnortEventMappingFactory {

    // flag detecting whether to skip ip6 addresses or not
    private final boolean mSkipIP6 = true;

    public SnortBarnyardFileEventMappingFactory(Properties props,
            ArrayList<HashMap<String, String>> data) {
        super(props, data);
    }

    // @Override
    protected void createMappingResult(Properties props, ArrayList<HashMap<String, String>> res) {
        super.initProperties(props);

        for (int i = 0; i < res.size(); i++) {
            EventType msgType = null;
            String discTime = null;

            // get classification <-> event message type
            Matcher classMatcher = Toolbox.getRegExPattern(Toolbox.REGEX_SNORT_ALERTLOG_CLASS)
                    .matcher(res.get(i).get("0"));
            if (classMatcher.find()) {
                msgType = getEventMappingForSignatureName(classMatcher.group().replace(
                        "Classification: ", ""));
            } else {
                // use default classification type
                msgType = EventType.other;
                IfMapClient.LOGGER.warning("could not find classification for entry[" + i
                        + "]...using default (other)");
            }

            // discovered time
            Matcher timestampMatcher = Toolbox.getRegExPattern(
                    Toolbox.REGEX_SNORT_ALERTLOG_TIMESTAMP).matcher(res.get(i).get("0"));
            if (timestampMatcher.find()) {
                discTime = Toolbox.getCurrentYear() + "/" + timestampMatcher.group();

                // check of current entry has a timestamp that was before the
                // IF-MAP-Client started
                if (!GeneralConfig.APPLICATION_MESSAGING_SENDOLD) {
                    if (!Toolbox.getCalendarFromString(rearrangeDate(discTime)).after(
                            Toolbox.getCalendarFromString(Toolbox.sClientStartTime))) {
                        discTime = null;
                    }
                }

            } else {
                IfMapClient.LOGGER.warning("could not find timestamp for entry[" + i
                        + "]...skipping entry!");
            }

            // check if event should be converted and send to server
            if (msgType != null && discTime != null && doConvert(msgType)) {
                EventMappingResult event = new EventMappingResult();

                // set event message-type
                event.setEventMessageType(msgType);

                // set discovered time
                event.setDiscoveredTime(convertDateToIFMAPDate(discTime));

                // check (and maybe set) source-ip
                Matcher ipMatcher = Toolbox.getRegExPattern(Toolbox.REGEX_GENERAL_IP4).matcher(
                        res.get(i).get("0"));
                if (ipMatcher.find()) {
                    event.setIp(ipMatcher.group());
                    event.setIpType("IPv4");
                } else {
                    // if no IPv4-Address, check for IPv6-address, in case that
                    // IPv6-Addresses should also be send to server
                    // TODO: this is currently detected by class-field. should
                    // be integrated into config-file soon(er or later)
                    if (!mSkipIP6) {
                        Matcher ip6Matcher = Toolbox.getRegExPattern(Toolbox.REGEX_GENERAL_IP6)
                                .matcher(res.get(i).get("0"));
                        if (ip6Matcher.find()) {
                            event.setIp(Toolbox
                                    .convertIP6AddressToIFMAPIP6AddressPattern(ip6Matcher.group()));
                            event.setIpType("IPv6");
                        }
                    }
                }

                // only continue if a source-ip was found
                if (event.getIp() != null) {

                    // set event name
                    Matcher typeMatcher = Toolbox
                            .getRegExPattern(Toolbox.REGEX_SNORT_ALERTLOG_TYPE).matcher(
                                    res.get(i).get("0"));
                    if (typeMatcher.find()) {
                        event.setName(typeMatcher.group().replace("] ", "").replace(" [", ""));
                    } else {
                        IfMapClient.LOGGER
                                .warning("could not find event name for entry[" + i + "]");
                    }

                    // priority
                    Matcher priorityMatcher = Toolbox.getRegExPattern(
                            Toolbox.REGEX_SNORT_ALERTLOG_PRIORITY).matcher(res.get(i).get("0"));
                    if (priorityMatcher.find()) {
                        event.setSignificance(getSignificanceValue(new Integer(priorityMatcher
                                .group().replace("Priority: ", "")).intValue()));
                    } else {
                        IfMapClient.LOGGER.warning("could not find priority for entry[" + i + "]");
                    }

                    // vulnerabilty-url
                    if (res.get(i).get("0") != null) {
                        Matcher vulnaribilityMatcher = Toolbox.getRegExPattern(
                                Toolbox.REGEX_SNORT_ALERTLOG_VULNARIBILITY_URI).matcher(
                                res.get(i).get("0"));
                        if (vulnaribilityMatcher.find()) {
                            event.setVulnerabilityUri(vulnaribilityMatcher.group()
                                    .replace("[Xref => ", "").replace("]h", " ; h")
                                    .replace("]", ""));
                            event.setVulnerabilityUri("http://cve.mitre.org/cgi-bin/cvename.cgi?name=2005-0068");
                        }
                    }

                    // discoverer id
                    event.setDiscovererId(SOAPMessageSender.getInstance().getIfMapPublisherId());

                    // dummy values for now...
                    event.setConfidence("100");
                    event.setMagnitude("45");

                    // add new event-result-entry to final result
                    super.mapResult.add(event);
                    IfMapClient.LOGGER.fine("mapped snort event has been added to result list: "
                            + event.showOnConsole());
                }
            }
        }
    }

    /**
     * rearrange passed in date (as parsed from alertlog-file) to fit IF-MAP
     * Timestamp format
     * 
     * @param String
     *            currentDate (YYYY/MM/DD-hh:mm:ss.S)
     * 
     * @return timestamp in IF-MAP format
     */
    private String rearrangeDate(String currentDate) {
        // YYYY/MM/DD-hh:mm:ss.S => [0]Date [1]Time
        String[] timestamp = currentDate.split("-");

        // YYYY/MM/DD => [0]Year [1] Month [2] Day
        String[] date = timestamp[0].split("/");

        // build new timestamp-string
        String newDate = date[0] + "-" + date[1] + "-" + date[2] + " " + timestamp[1];

        // make timestamp-string combatible with ifmap-specification
        return newDate.substring(0, newDate.length() - 7);
    }

    /**
     * convert passed in string to string in IFMAP-Timestamp format
     * 
     * @param String
     *            currentDate (YYYY/MM/DD-hh:mm:ss.S)
     * 
     * @return timestamp in IF-MAP format
     */

    private String convertDateToIFMAPDate(String currentDate) {
        // YYYY/MM/DD-hh:mm:ss.S => [0]Date [1]Time
        String[] timestamp = currentDate.split("-");

        // YYYY/MM/DD => [0]Year [1] Month [2] Day
        String[] date = timestamp[0].split("/");

        // fomrat according to ifmap-specification
        String newDate = date[0] + "-" + date[1] + "-" + date[2] + "T" + timestamp[1];
        return newDate.substring(0, newDate.length() - 7) + "Z";
    }
}