/* 
 * IPTablesEventMappingFactory.java        0.1.4 12/02/16
 *
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.mappingfactory;

import de.esukom.decoit.ifmapclient.config.GeneralConfig;
import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.messaging.SOAPMessageSender;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import de.fhhannover.inform.trust.ifmapj.metadata.EventType;
import de.fhhannover.inform.trust.ifmapj.metadata.Significance;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Matcher;

/**
 * concrete implementation of abstract mapping-factory for mapping values from
 * polling-threads to event objects that can be send to MAP-Server
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class IPTablesEventMappingFactory extends MappingFactory {

    // properties from mapping config file
    private EventType mEventtypeDefault;
    private String mMagnitudeDefault;
    private String mConfidenceDefault;
    private Significance mSignificanceDefault;
    private int mDuplicateEntriesTimeDelay;

    /**
     * constructor
     * 
     * @param props
     *            properties object
     * @param data
     *            List of Hash-Maps containing data from file poller
     */
    public IPTablesEventMappingFactory(Properties props, ArrayList<HashMap<String, String>> data) {
        super(props, data);
    }

    @Override
    protected void initProperties(Properties props) {
        boolean criticalErrorOcured = false;

        // event to be processed/mapped
        mEventtypeDefault = EventType.behavioralChange; // default!

        String eventTypeFromConfig = Toolbox.getStringProperty("iptables.eventmapping.eventtype",
                props, false);
        if (mEventtypeDefault == null) {
            criticalErrorOcured = true;
        }

        // map event-type-string to ifmaplibj EventType-type
        if (eventTypeFromConfig == "botnet-infection") {
            mEventtypeDefault = EventType.botnetInfection;

        } else if (eventTypeFromConfig == "cve") {
            mEventtypeDefault = EventType.cve;

        } else if (eventTypeFromConfig == "excessive-flows") {
            mEventtypeDefault = EventType.excessiveFlows;

        } else if (eventTypeFromConfig == "other") {
            mEventtypeDefault = EventType.other;

        } else if (eventTypeFromConfig == "p2p") {
            mEventtypeDefault = EventType.p2p;

        } else if (eventTypeFromConfig == "policy-violation") {
            mEventtypeDefault = EventType.policyViolation;

        } else if (eventTypeFromConfig == "worm-infection") {
            mEventtypeDefault = EventType.wormInfection;

        } else {
            IfMapClient.LOGGER
                    .config("could not load default event-type value from iptables mapping config...using default (behavioral-change)");
        }

        // magnitude mapping
        mMagnitudeDefault = Toolbox.getStringProperty("iptables.eventmapping.magnitude", props,
                false);
        if (mMagnitudeDefault == null) {
            criticalErrorOcured = true;
        }

        // confidence mapping
        mConfidenceDefault = Toolbox.getStringProperty("iptables.eventmapping.confidence", props,
                false);
        if (mConfidenceDefault == null) {
            criticalErrorOcured = true;
        }

        // significance mapping
        mSignificanceDefault = Significance.informational; // default!
        String significanceFromConfig = Toolbox.getStringProperty(
                "iptables.eventmapping.significance", props, false);

        // map significance-string to ifmaplibj significance datatype
        if (significanceFromConfig == "important") {
            mSignificanceDefault = Significance.important;

        } else if (significanceFromConfig == "critical") {
            mSignificanceDefault = Significance.critical;

        } else {
            IfMapClient.LOGGER
                    .config("could not load default significance value from iptables mapping config...using default (informational)");
        }

        // time delay for events (with the samen name and address) to be
        // considered equal
        mDuplicateEntriesTimeDelay = Toolbox.getIntPropertyWithDefault(
                "iptables.eventmapping.timedelay", 0, props, false);

        if (criticalErrorOcured) {
            IfMapClient.exit("error while initializing iptables-EventMappingFactory");
        }
    }

    /**
     * map result-set from database-query to a list of attribute-value-pairs
     * (realized as ArrayList of HashMaps for now)
     * 
     * @param resultSet
     *            result-set of database query
     * 
     * @return ArrayList list containing the DB-Entries as HashMaps
     */
    // @Override
    protected void createMappingResult(Properties props, ArrayList<HashMap<String, String>> res) {

        initProperties(props);

        // Needed for comparing potential duplicate entries in result?
        boolean isFirstEntry = true;
        String prevDiscTime = null;

        for (int i = 0; i < res.size(); i++) {

            String discTime = null;

            EventMappingResult event = new EventMappingResult();

            // timestamp -> discovered time
            Matcher timestampMatcher = Toolbox.getRegExPattern(
                    Toolbox.REGEX_IPTABLES_IFMAP_TIMESTAMP).matcher(res.get(i).get("0"));
            if (timestampMatcher != null && timestampMatcher.find()) {
                discTime = timestampMatcher.group();
            } else {
                IfMapClient.LOGGER.warning("could not find timestamp for entry[" + i
                        + "]...skipping entry");
                continue;
            }

            // get source and destination-ip and append it to event-name
            String srcIp, dstIp = null;
            Matcher srcMatcher = Toolbox.getRegExPattern("SRC=" + Toolbox.REGEX_GENERAL_IP4)
                    .matcher(res.get(i).get("0"));
            if (srcMatcher != null && srcMatcher.find()) {
                srcIp = srcMatcher.group().replace("SRC=", "");
            } else {
                IfMapClient.LOGGER.warning("could not find source ip for entry[" + i
                        + "]...skipping entry");
                continue;
            }
            event.setIp(srcIp);

            Matcher dstMatcher = Toolbox.getRegExPattern("DST=" + Toolbox.REGEX_GENERAL_IP4)
                    .matcher(res.get(i).get("0"));
            if (dstMatcher != null && dstMatcher.find()) {
                dstIp = dstMatcher.group().replace("DST=", "");
                ;
            } else {
                IfMapClient.LOGGER.warning("could not find destination ip for entry[" + i
                        + "]...using default (not detected)");
                dstIp = "not detected";
            }

            // do not publish datastreams-detected-Events between MAP-Server and
            // IPTables-Client!
            if (srcIp.equals(IfMapClient.sMapServerIP) & dstIp.equals(IfMapClient.sClientIP)
                    || srcIp.equals(IfMapClient.sClientIP) & dstIp.equals(IfMapClient.sMapServerIP)) {
                IfMapClient.LOGGER.fine("Event is Datastream  between Server and Client (src="
                        + srcIp + ",dst=" + dstIp + "...not publishing Event!");
                continue;
            }

            // set event name
            event.setName("datastream detected from " + srcIp + " to " + dstIp);

            // check if current entry equals last entry, if so throw it away
            // because its duplicate. this is done by
            // 1) checking for equal name and ip
            // 2) if both matches, check if current event happened after the
            // event that is already inside result list
            boolean insertEntry = false;
            if (!isFirstEntry) {
                for (int j = 0; j < super.mapResult.size(); j++) {
                    EventMappingResult tempEvent = (EventMappingResult) super.mapResult.get(j);
                    // check if current temporary EventMappingResult-object has
                    // the same name and ip as the current
                    // EventMappingResult-object
                    if (tempEvent.getName().equals(event.getName())
                            & tempEvent.getIp().equals(event.getIp())) {
                        // compare time-stamps of both objects
                        Calendar prevCalendar = Toolbox.getCalendarFromString(prevDiscTime);
                        prevCalendar.add(Calendar.SECOND, mDuplicateEntriesTimeDelay * (-1));
                        if (Toolbox.getCalendarFromString(discTime).after(prevCalendar)) {
                            super.mapResult.remove(j);
                            insertEntry = true;
                        } else {
                            prevDiscTime = discTime;
                        }
                    } else {
                        insertEntry = true;
                    }
                }
            } else {
                isFirstEntry = false;
                insertEntry = true;

            }

            // generate "datastream detected" - event
            if (insertEntry) {
                event.setDiscovererId(SOAPMessageSender.getInstance().getIfMapPublisherId());

                // set predefined mapping-values from config
                event.setSignificance(mSignificanceDefault);
                event.setEventMessageType(mEventtypeDefault);
                event.setConfidence(mConfidenceDefault);
                event.setMagnitude(mMagnitudeDefault);
                event.setIpType("IPv4");

                // set time (and store it for date-comparison in next loop
                prevDiscTime = discTime;
                event.setDiscoveredTime(convertDateToIFMAPDate(discTime));

                // add event
                
                super.mapResult.add(event);
                IfMapClient.LOGGER.fine("mapped iptables event has been added to result list: "
                        + event.showOnConsole());
            }
        }
    }

    /**
     * convert passed in string to string in IFMAP-Timestamp format
     * 
     * @param String
     *            currentDate (YYYY/MM/DD-hh:mm:ss.S)
     * 
     * @return timestamp in IF-MAP format
     */

    private String convertDateToIFMAPDate(String currentDate) {
        // YYYY/MM/DD-hh:mm:ss.S => [0]Date [1]Time
        String[] timestamp = currentDate.split(" ");

        // YYYY/MM/DD => [0]Year [1] Month [2] Day
        String[] date = timestamp[0].split("-");

        // fomrat according to ifmap-specification
        return date[0] + "-" + date[1] + "-" + date[2] + "T" + timestamp[1] + "Z";
    }
}